/**
 * Scrolls a page to the bottom to load all content
 * @param {Page} page - Puppeteer page object
 * @param {number} duration - Milliseconds between scroll steps (default: 100, lower = faster)
 * @returns {Promise} - Resolves when scrolling is complete
 */
async function scrollBrollPageToBottom(page, duration = 100) {
  return page.evaluate((duration) => {
    return new Promise((resolve) => {
      let scrolled = 0;
      let distance = 100;

      const timer = setInterval(() => {
        window.scrollBy(0, distance);
        scrolled += distance;

        if (scrolled >= document.body.scrollHeight - window.innerHeight) {
          clearInterval(timer);
          resolve();
        }
      }, duration);
    });
  }, duration);
}

/**
 * Extracts data from elements on a page according to configuration
 * @param {Page} page - Puppeteer page object
 * @param {object} instruction - Configuration for data extraction
 * @returns {Promise<array>} - Array of extracted objects
 */
async function scrapeBrollResults(page, instruction) {
  // Forward console logs from browser to Node
  page.on('console', (msg) => console.log('BROWSER:', msg.text()));

  return page.evaluate((config) => {
    console.log(`Finding elements: ${config.resultSelector}`);
    const items = document.querySelectorAll(config.resultSelector);
    console.log(`Found ${items.length} results`);

    return Array.from(items).map((el) => {
      const data = {};

      if (Array.isArray(config.attributesToExtract)) {
        config.attributesToExtract.forEach((attr) => {
          const target = attr.selector ? el.querySelector(attr.selector) : el;
          if (!target) return;

          let val = null;
          if (attr.attribute === 'text' || attr.attribute === 'textContent') {
            val = target.textContent.trim();
          } else if (attr.attribute.startsWith('dataset.')) {
            const key = attr.attribute.split('.')[1];
            val = target.dataset[key];
          } else if (attr.name === 'thumbnail') {
            // Clean up thumbnail URLs
            val = target.getAttribute(attr.attribute).split(' ')[0];
            val = val.split('%20')[0];
          } else {
            val = target.getAttribute(attr.attribute);
          }

          data[attr.name] = val;
        });
      }

      return data;
    });
  }, instruction);
}

/**
 * Checks if a user is logged in by looking for a specific element on the page
 * @param {Page} page - Puppeteer page object
 * @param {string} selector - CSS selector to check for
 * @returns {Promise<boolean>} - Whether the element was found (logged in)
 */
async function checkForLoggedInSelector(page, selector) {
  return page
    .evaluate((selector) => {
      console.warn('Checking for element with selector:', selector);
      const element = document.querySelector(selector);
      console.warn('Element found:', element !== null);
      return element !== null;
    }, selector)
    .catch((err) => {
      console.error('Error checking login status:', err);
      return false;
    });
}
